/*
 * linux/arch/arm/mach-uniphier/cache.c
 *
 * Copyright (C) 2011 Panasonic Corporation
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/hardirq.h>
#include <linux/bootmem.h>

#include <asm/cacheflush.h>
#include <asm/pgalloc.h>
#include <asm/setup.h>
#include <mach/hardware.h>

#define SIZE_ALL_CACHE		0x10000000	/* Used by purge_cache,when the parameter size of purge_cache is 
				  		   set as CACHE_ALL, the operation is done on the whole cache */

#define MAXAREAPURGE_L1C	0x00010000	/* 64KB */
#define MAXAREAPURGE_L2C	0x00100000	/* 1MB */
#if MAXAREAPURGE_L1C > MAXAREAPURGE_L2C
#error MAXAREAPURGE_L1C must be smaller than MAXAREAPURGE_L2C
#endif

#define UNIPHIER_PURGE_INVALID	0xffffffff

static int __mn_addr_trans(unsigned long vaddr, unsigned long *paddr)
{
	unsigned long flags, par;
	local_irq_save(flags);

	__asm__ __volatile__(
"	mcr	p15, 0, %1, c7, c8, 0\n"
"	mrc	p15, 0, %0, c7, c4, 0\n"
	: "=&r" (par)
	: "r" (vaddr)
	: "cc");

	local_irq_restore(flags);
	if (likely((par & 1) == 0)) {
		if (par & 2) {
			par = (par & SUPERSECTION_MASK) | (vaddr & ~SUPERSECTION_MASK);
		} else {
			par = (par & PAGE_MASK) | (vaddr & ~PAGE_MASK);
		}
	} else {
		/* fixup for vmalloc area or old page (i.e. not young page) */
		pgd_t *pgd;
		pud_t *pud;
		pmd_t *pmd;
		pte_t *pte;

		if (vaddr >= TASK_SIZE) {
			pgd = pgd_offset_k(vaddr);
		} else {
			pgd = cpu_get_pgd() + pgd_index(vaddr);
		}

		if (pgd_none(*pgd)) {
			return -1;
		}
		pud = pud_offset(pgd, vaddr);
		if (pud_none(*pud)) {
			return -1;
		}
		pmd = pmd_offset(pud, vaddr);
		if (pmd_none(*pmd)) {
			return -1;
		}
		if (unlikely((pmd_val(*pmd) & PMD_TYPE_MASK) == PMD_TYPE_SECT)) {
			if (pmd_val(*pmd) & PMD_SECT_SUPER) {
				par = (pmd_val(*pmd) & SUPERSECTION_MASK)
					 | (vaddr & ~SUPERSECTION_MASK);
			}
			else {
				par = (pmd_val(*pmd) & SECTION_MASK)
					 | (vaddr & ~SECTION_MASK);
			}
		}
		else if (likely((pmd_val(*pmd) & PMD_TYPE_MASK) == PMD_TYPE_TABLE)) {
			pte = pte_offset_map(pmd, vaddr);
			if (!pte_present(*pte)) {
				return -1;
			}
			par = (pte_val(*pte) & PAGE_MASK)
				 | (vaddr & ~PAGE_MASK);
			pte_unmap(pte);
		}
		else {
			return -1;
		}
	}
	*paddr = par;
	return 0;
}

unsigned long mn_addr_trans(unsigned long vaddr)
{
	unsigned long paddr = 0;
	if (__mn_addr_trans(vaddr, &paddr) != 0) {
		return 0;
	}
	return paddr;
}
EXPORT_SYMBOL(mn_addr_trans);

struct uniphier_purge_area {
	unsigned long vaddr;
	unsigned long pfn;
	pte_t *pte[UNIPHIER_PURGE_AREA_SIZE/PAGE_SIZE];
};

DEFINE_PER_CPU(struct uniphier_purge_area, uniphier_purge_area);

static inline unsigned long
uniphier_map_purge_area(unsigned long pfn, unsigned long *old_pfn)
{
	struct uniphier_purge_area *purge_area;
	int i;
	unsigned long vaddr, flags;

	purge_area = &get_cpu_var(uniphier_purge_area);
	*old_pfn = purge_area->pfn;

	if (pfn == purge_area->pfn) {
		return purge_area->vaddr;
	}

	local_irq_save(flags);
	purge_area->pfn = pfn;
	vaddr = purge_area->vaddr;
	for (i = 0;i < UNIPHIER_PURGE_AREA_SIZE/PAGE_SIZE;i++, pfn++) {
		set_pte_ext(purge_area->pte[i],
			 pfn_pte(pfn, PAGE_KERNEL), PTE_EXT_TEX(1));
		flush_tlb_kernel_page(vaddr);
		vaddr += PAGE_SIZE;
	}
	local_irq_restore(flags);
	return purge_area->vaddr;
}

static inline void uniphier_unmap_purge_area(unsigned long old_pfn) {
	struct uniphier_purge_area *purge_area;
	int i;
	unsigned long vaddr, flags;

	purge_area = &__get_cpu_var(uniphier_purge_area);

	if (likely(!in_interrupt()) || purge_area->pfn == old_pfn) {
		goto skip;
	}

	local_irq_save(flags);
	purge_area->pfn = old_pfn;
	vaddr = purge_area->vaddr;
	for (i = 0;i < UNIPHIER_PURGE_AREA_SIZE/PAGE_SIZE;i++, old_pfn++) {
		set_pte_ext(purge_area->pte[i],
			 pfn_pte(old_pfn, PAGE_KERNEL), PTE_EXT_TEX(1));
		flush_tlb_kernel_page(vaddr);
		vaddr += PAGE_SIZE;
	}
	local_irq_restore(flags);

skip:
	put_cpu_var(uniphier_purge_area);
}

static int __init uniphier_init_purge_area(void) {
	struct uniphier_purge_area *purge_area;
	unsigned long vaddr, size;
	pgd_t *pgd;
	pmd_t *pmd;
	pte_t *pte, **pte_p;
	int i, j;

	vaddr = UNIPHIER_PURGE_AREA_BASE;
	for (i = 0;i < NR_CPUS;i++) {
		purge_area = &per_cpu(uniphier_purge_area, i);
		purge_area->vaddr = vaddr;
		purge_area->pfn = UNIPHIER_PURGE_INVALID;
		pte_p = &(purge_area->pte[0]);
		size = UNIPHIER_PURGE_AREA_SIZE;
		while (size) {
			pgd = pgd_offset_k(vaddr);
			pmd = pmd_alloc(&init_mm, pgd, vaddr);
			BUG_ON(!pmd);
			pte = pte_alloc_kernel(pmd, vaddr);
			BUG_ON(!pte);
			for (j = 0;j < PTRS_PER_PTE && size;j++) {
				*(pte_p++) = pte_offset_kernel(pmd, vaddr);
				vaddr += PAGE_SIZE;
				size -= PAGE_SIZE;
			}
		}
	}
	return 0;
}
early_initcall(uniphier_init_purge_area);

#ifdef CONFIG_SMP
static void uniphier_smp_clean_dcache_all(void *info) {
	clean_dcache_all();
}

static void uniphier_smp_flush_dcache_all(void *info) {
	flush_dcache_all();
}
#endif /* CONFIG_SMP */

static void uniphier_clean_dcache_all(void) {
#ifdef CONFIG_SMP
	preempt_disable();
#endif
	clean_dcache_all();
#ifdef CONFIG_SMP
	smp_call_function(uniphier_smp_clean_dcache_all, NULL, 1);
	preempt_enable();
#endif
}

static void uniphier_flush_dcache_all(void) {
#ifdef CONFIG_SMP
	preempt_disable();
#endif
	flush_dcache_all();
#ifdef CONFIG_SMP
	smp_call_function(uniphier_smp_flush_dcache_all, NULL, 1);
	preempt_enable();
#endif
}

static void uniphier_flush_cache_all(void) {
#ifdef CONFIG_SMP
	preempt_disable();
	smp_call_function(uniphier_smp_flush_dcache_all, NULL, 1);
#endif
	flush_cache_all();
#ifdef CONFIG_SMP
	preempt_enable();
#endif
}

void uniphier_dmac_clean_range(const void *start, const void *end) {
	unsigned long size = (unsigned long)end - (unsigned long)start;
	if (size >= MAXAREAPURGE_L1C && !irqs_disabled() && !in_interrupt()) {
		uniphier_clean_dcache_all();
	}
	else {
		dmac_clean_range(start, end);
	}
}

void uniphier_dmac_inv_range(const void *start, const void *end) {
	unsigned long size = (unsigned long)end - (unsigned long)start;
	if (size >= MAXAREAPURGE_L1C && !irqs_disabled() && !in_interrupt()) {
		uniphier_flush_dcache_all();
	}
	else {
		dmac_inv_range(start, end);
	}
}

void uniphier_dmac_flush_range(const void *start, const void *end) {
	unsigned long size = (unsigned long)end - (unsigned long)start;
	if (size >= MAXAREAPURGE_L1C && !irqs_disabled() && !in_interrupt()) {
		uniphier_flush_dcache_all();
	}
	else {
		dmac_flush_range(start, end);
	}
}

static void purge_L1cache_all(int flag) {
	switch (flag) {
	case PURGE_CACHE_D_PURGE:
		uniphier_clean_dcache_all();
		break;
	case PURGE_CACHE_D_PURGE_INV:
		uniphier_flush_dcache_all();
		break;
	case PURGE_CACHE_D_INV:
		uniphier_flush_dcache_all();
		break;
	default: /* PURGE_CACHE_ID_SYNC || PURGE_CACHE_I_INV */
		uniphier_flush_cache_all();
		break;
	}
}

/* purge_L1cache has same specification to purge_cache */

void purge_L1cache(unsigned long start, unsigned long size, int flag)
{
	unsigned long pfn, vaddr, offset, tmp_size, old_pfn;

	if (unlikely(size >= MAXAREAPURGE_L1C)) {
		if (!irqs_disabled() && !in_interrupt()) {
			purge_L1cache_all(flag);
			return;
		}
		else if (size == SIZE_ALL_CACHE) {
			printk(KERN_ERR
				"BUG: Cannot purge the entire cache while"
				" irqs disabled or in interrupt context.\n");
			printk(KERN_ERR
				"in_interrupt(): %lu, irqs_disabled(): %d\n",
				in_interrupt(), irqs_disabled());
		}
	}

	pfn = (start & (~(UNIPHIER_PURGE_AREA_SIZE - 1))) >> PAGE_SHIFT;
	offset = start & (UNIPHIER_PURGE_AREA_SIZE - 1);

	while (size) {
		tmp_size = (size > (UNIPHIER_PURGE_AREA_SIZE - offset))
			 	? (UNIPHIER_PURGE_AREA_SIZE - offset): size;
		vaddr = uniphier_map_purge_area(pfn, &old_pfn);
		vaddr += offset;
		switch (flag) {
		case PURGE_CACHE_D_PURGE:
			dmac_clean_range((void *)vaddr, (void *)(vaddr + tmp_size));
			break;
		case PURGE_CACHE_D_PURGE_INV:
			dmac_flush_range((void *)vaddr, (void *)(vaddr + tmp_size));
			break;
		case PURGE_CACHE_D_INV:
			dmac_inv_range((void *)vaddr, (void *)(vaddr + tmp_size));
			break;
		default: /* PURGE_CACHE_I_INV or PURGE_CACHE_ID_SYNC */
			dmac_clean_range((void *)vaddr, (void *)(vaddr + tmp_size));
			flush_icache_all();
			break;
		}
		uniphier_unmap_purge_area(old_pfn);
		offset = 0;
		pfn += UNIPHIER_PURGE_AREA_SIZE >> PAGE_SHIFT;
		size -= tmp_size;
	}
	return;
}
EXPORT_SYMBOL(purge_L1cache);

static void __vaddr_purge_L1cache(unsigned long start, unsigned long size, int flag)
{
	switch (flag) {
	case PURGE_CACHE_D_PURGE:
		dmac_clean_range((void *)start, (void *)(start + size));
		break;
	case PURGE_CACHE_D_PURGE_INV:
		dmac_flush_range((void *)start, (void *)(start + size));
		break;
	case PURGE_CACHE_D_INV:
		dmac_inv_range((void *)start, (void *)(start + size));
		break;
	default: /* PURGE_CACHE_I_INV or PURGE_CACHE_ID_SYNC */
		dmac_clean_range((void *)start, (void *)(start + size));
		flush_icache_all();
		break;
	}
	return;
}

void
purge_L2cache(unsigned long start, unsigned long size, int flag)
{
#ifdef CONFIG_OUTER_CACHE
	unsigned long end = start + size;
	if (end <= start)	return;

	if (flag == PURGE_CACHE_D_PURGE) {
		outer_clean_range(start, end);
	}
	else if (flag == PURGE_CACHE_D_PURGE_INV) {
		outer_flush_range(start, end);
	}else if (flag == PURGE_CACHE_D_INV){
		outer_inv_range(start, end);
	}
#endif /* CONFIG_OUTER_CACHE */
}
EXPORT_SYMBOL(purge_L2cache);

void purge_cache(unsigned long start, unsigned long size, int flag)
{
	purge_L1cache(start, size, flag);
	purge_L2cache(start, size, flag);
}
EXPORT_SYMBOL(purge_cache);

static void
__vaddr_purge_cache(unsigned long vstart, unsigned long size, int flag, int l2)
{
	unsigned long w, paddr, vend;
	int l1_all = 0;
#ifdef CONFIG_OUTER_CACHE
	unsigned long w_post = 0, paddr_post = 0;
	int postponed = 0;
#endif /* CONFIG_OUTER_CACHE */

	if (unlikely(size >= MAXAREAPURGE_L1C)) {
		if (!irqs_disabled() && !in_interrupt()) {
			purge_L1cache_all(flag);
			if (l2) {
				l1_all = 1;
			} else {
				return;
			}
		}
		else if (size == SIZE_ALL_CACHE) {
			printk(KERN_ERR
				"BUG: Cannot purge the entire cache while"
				" irqs disabled or in interrupt context.\n");
			printk(KERN_ERR
				"in_interrupt(): %lu, irqs_disabled(): %d\n",
				in_interrupt(), irqs_disabled());
		}
	}

	for (vend = vstart + size; vstart < vend; vstart += w) {
		w = PAGE_SIZE - (vstart & ~PAGE_MASK);
		if ((vend - vstart) < w) {
			w = vend - vstart;
		}
		if (unlikely(__mn_addr_trans(vstart, &paddr) != 0)) {
			continue;
		}
		if (!l1_all) {
			if (vstart < TASK_SIZE) {
				/* avoid fault for old page */
				purge_L1cache(paddr, w, flag);
			} else {
				__vaddr_purge_L1cache(vstart, w, flag);
			}
		}
		if (l2) {
#ifdef CONFIG_OUTER_CACHE
			if (postponed) {
				if (paddr == paddr_post + w_post) {
					w_post += w;
					continue;
				}
				purge_L2cache(paddr_post, w_post, flag);
			}
			paddr_post = paddr;
			w_post = w;
			postponed = 1;
#endif /* CONFIG_OUTER_CACHE */
		}
	}
#ifdef CONFIG_OUTER_CACHE
	if (postponed) {
		purge_L2cache(paddr_post, w_post, flag);
	}
#endif /* CONFIG_OUTER_CACHE */
}

void vaddr_purge_L1cache(unsigned long vstart, unsigned long size, int flag)
{
	__vaddr_purge_cache(vstart, size, flag, 0);
}
EXPORT_SYMBOL(vaddr_purge_L1cache);

void vaddr_purge_cache(unsigned long vstart, unsigned long size, int flag)
{
	__vaddr_purge_cache(vstart, size, flag, 1);
}
EXPORT_SYMBOL(vaddr_purge_cache);

void write_buffer_drain(void)
{
	u32 tmp;
	dsb();
#ifdef CONFIG_OUTER_CACHE
	outer_sync();
#endif /* CONFIG_OUTER_CACHE */
	tmp = __raw_readl(UNIPHIER_KERNEL_UNCACHE_BASE);
}
EXPORT_SYMBOL(write_buffer_drain);
